class DarkModeToggle {

    constructor() {
        this.darkModeEnabled = false;
        this.toggle = null;
        this.container = null;

        this.STORAGE_KEY = 'darkModePreference';

        this.init();
    }

    init() {

        this.darkModeEnabled = this.getInitialState();

        this.applyDarkMode(this.darkModeEnabled);

        this.createFloatingToggle();

        this.watchSystemPreference();
    }

    getInitialState() {

        const savedPreference = localStorage.getItem(this.STORAGE_KEY);

        if (savedPreference !== null) {
            return savedPreference === 'true';
        }

        if (window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches) {
            return true;
        }

        return false;
    }

    createFloatingToggle() {

        this.container = document.createElement('div');
        this.container.className = 'floating-dark-mode-toggle';

        const buttonWrapper = document.createElement('div');
        buttonWrapper.className = 'dark-mode-toggle-btn';

        const icon = document.createElement('span');
        icon.className = 'dark-mode-icon';
        this.iconElement = icon;

        const text = document.createElement('span');
        text.className = 'dark-mode-text';
        text.textContent = this.darkModeEnabled ? 'Dark Mode' : 'Light Mode';
        this.textElement = text;

        this.toggle = new Toggle({
            label: '',
            initialState: this.darkModeEnabled,
            iconOn: '🌙',
            iconOff: '☀️',
            onChange: (state) => {
                this.setDarkMode(state);
            }
        });

        buttonWrapper.appendChild(icon);
        buttonWrapper.appendChild(this.toggle.getElement());
        buttonWrapper.appendChild(text);

        this.container.appendChild(buttonWrapper);

        document.body.appendChild(this.container);
    }

    applyDarkMode(enabled) {
        if (enabled) {
            document.documentElement.classList.add('dark-mode');
        } else {
            document.documentElement.classList.remove('dark-mode');
        }
    }

    setDarkMode(enabled) {
        this.darkModeEnabled = enabled;

        this.applyDarkMode(enabled);

        localStorage.setItem(this.STORAGE_KEY, enabled.toString());

        this.updateUI();
    }

    updateUI() {
        if (this.textElement) {
            this.textElement.textContent = this.darkModeEnabled ? 'Dark Mode' : 'Light Mode';
        }
    }

    toggleDarkMode() {
        this.setDarkMode(!this.darkModeEnabled);
    }

    watchSystemPreference() {
        if (window.matchMedia) {
            const mediaQuery = window.matchMedia('(prefers-color-scheme: dark)');

            mediaQuery.addEventListener('change', (e) => {

                const savedPreference = localStorage.getItem(this.STORAGE_KEY);
                if (savedPreference === null) {
                    this.setDarkMode(e.matches);
                    if (this.toggle) {
                        this.toggle.setState(e.matches);
                    }
                }
            });
        }
    }

    isDarkMode() {
        return this.darkModeEnabled;
    }

    clearPreference() {
        localStorage.removeItem(this.STORAGE_KEY);
        const systemPreference = window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches;
        this.setDarkMode(systemPreference);
        if (this.toggle) {
            this.toggle.setState(systemPreference);
        }
    }

    destroy() {
        if (this.toggle) {
            this.toggle.destroy();
        }
        if (this.container && this.container.parentNode) {
            this.container.parentNode.removeChild(this.container);
        }
        this.container = null;
        this.toggle = null;
    }
}

if (typeof document !== 'undefined') {
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => {
            window.darkModeToggle = new DarkModeToggle();
        });
    } else {
        window.darkModeToggle = new DarkModeToggle();
    }
}

if (typeof module !== 'undefined' && module.exports) {
    module.exports = DarkModeToggle;
}