class Button {
    constructor(options = {}) {
        this.options = {
            containerId: options.containerId || null,
            id: options.id || this.generateId(),
            text: options.text || 'Button',
            variant: options.variant || 'primary',
            size: options.size || 'medium',
            disabled: options.disabled || false,
            loading: options.loading || false,
            type: options.type || 'button',
            onClick: options.onClick || null,
            className: options.className || '',
            fullWidth: options.fullWidth || false,
            ...options
        };

        this.container = null;
        this.buttonElement = null;
        this.isLoading = this.options.loading;
        this.isDisabled = this.options.disabled;

        this.init();
    }

    generateId() {
        return 'button-' + Math.random().toString(36).substring(2, 11);
    }

    init() {
        if (this.options.containerId) {
            this.container = document.getElementById(this.options.containerId);
            if (!this.container) {
                console.error(`Button container with ID "${this.options.containerId}" not found`);
                return;
            }
        }

        this.render();
        this.attachEventListeners();
    }

    render() {
        const buttonClasses = this.getButtonClasses();
        const loadingHtml = this.isLoading ? '<span class="loading"></span>' : '';

        const buttonHtml = `
            <button
                type="${this.options.type}"
                id="${this.options.id}"
                class="${buttonClasses}"
                ${this.isDisabled || this.isLoading ? 'disabled' : ''}
                aria-label="${this.options.text}"
            >
                ${this.isLoading ? loadingHtml : ''}
                <span class="button-text">${this.options.text}</span>
            </button>
        `;

        if (this.container) {
            this.container.innerHTML = buttonHtml;
            this.buttonElement = this.container.querySelector('button');
        } else {

            return buttonHtml;
        }
    }

    getButtonClasses() {
        const classes = ['btn', `btn-${this.options.variant}`];

        if (this.options.size !== 'medium') {
            classes.push(`btn-${this.options.size}`);
        }

        if (this.options.fullWidth) {
            classes.push('btn-full-width');
        }

        if (this.isLoading) {
            classes.push('btn-loading');
        }

        if (this.options.className) {
            classes.push(this.options.className);
        }

        return classes.join(' ');
    }

    attachEventListeners() {
        if (!this.buttonElement) return;

        this.buttonElement.addEventListener('click', (e) => {
            if (this.isDisabled || this.isLoading) {
                e.preventDefault();
                return;
            }

            if (this.options.onClick && typeof this.options.onClick === 'function') {
                this.options.onClick(e, this);
            }
        });
    }

    setLoading(loading = true) {
        this.isLoading = loading;
        this.updateButton();
    }

    setDisabled(disabled = true) {
        this.isDisabled = disabled;
        this.updateButton();
    }

    setText(text) {
        this.options.text = text;
        if (this.buttonElement) {
            const textElement = this.buttonElement.querySelector('.button-text');
            if (textElement) {
                textElement.textContent = text;
            }
        }
    }

    setVariant(variant) {
        if (this.buttonElement) {

            this.buttonElement.classList.remove(`btn-${this.options.variant}`);

            this.options.variant = variant;
            this.buttonElement.classList.add(`btn-${variant}`);
        }
    }

    updateButton() {
        if (!this.buttonElement) return;

        if (this.isDisabled || this.isLoading) {
            this.buttonElement.setAttribute('disabled', '');
        } else {
            this.buttonElement.removeAttribute('disabled');
        }

        this.buttonElement.className = this.getButtonClasses();

        const loadingHtml = this.isLoading ? '<span class="loading"></span>' : '';

        this.buttonElement.innerHTML = `
            ${this.isLoading ? loadingHtml : ''}
            <span class="button-text">${this.options.text}</span>
        `;
    }

    getElement() {
        return this.buttonElement;
    }

    destroy() {
        if (this.buttonElement) {
            this.buttonElement.remove();
        }
        if (this.container) {
            this.container.innerHTML = '';
        }
    }
}

if (typeof module !== 'undefined' && module.exports) {
    module.exports = Button;
}