class ImageUpload {
    constructor(options = {}) {
        this.options = {
            containerId: options.containerId || null,
            id: options.id || this.generateId(),
            name: options.name || options.id || this.generateId(),
            label: options.label || 'Upload Images',
            required: options.required || false,
            maxFiles: options.maxFiles || 10,
            maxFileSize: options.maxFileSize || 5 * 1024 * 1024,
            acceptedFormats: options.acceptedFormats || ['image/jpeg', 'image/jpg', 'image/png'],
            onChange: options.onChange || null,
            helpText: options.helpText || null,
            ...options
        };

        this.container = null;
        this.fileInput = null;
        this.previewContainer = null;
        this.errorElement = null;
        this.images = [];
        this.isValid = true;

        this.init();
    }

    generateId() {
        return 'image-upload-' + Math.random().toString(36).substr(2, 9);
    }

    init() {
        if (this.options.containerId) {
            this.container = document.getElementById(this.options.containerId);
            if (!this.container) {
                console.error(`ImageUpload: Container with ID "${this.options.containerId}" not found`);
                return;
            }
            this.render();
        }
    }

    render() {
        if (!this.container) return;

        const formGroupClass = `form-group image-upload-component`;

        this.container.innerHTML = `
            <div class="${formGroupClass}">
                <label>${this.options.label}${this.options.required ? ' *' : ''}</label>
                <div class="image-upload-input-wrapper">
                    <input
                        type="file"
                        id="${this.options.id}"
                        name="${this.options.name}"
                        multiple
                        accept="${this.options.acceptedFormats.join(',')}"
                        class="image-upload-input"
                        style="display: none;"
                    />
                    <label for="${this.options.id}" class="image-upload-button">
                        <span class="upload-icon">📁</span>
                        <span class="upload-text">Click to upload or drag and drop</span>
                        <span class="upload-hint">JPEG, JPG, PNG up to ${this.formatFileSize(this.options.maxFileSize)}</span>
                    </label>
                </div>
                <div class="image-upload-preview-container"></div>
                ${this.options.helpText ? `<div class="image-upload-help-text">${this.options.helpText}</div>` : ''}
                <div class="image-upload-error" style="display: none;"></div>
            </div>
        `;

        this.fileInput = this.container.querySelector(`#${this.options.id}`);
        this.previewContainer = this.container.querySelector('.image-upload-preview-container');
        this.errorElement = this.container.querySelector('.image-upload-error');
        const uploadButton = this.container.querySelector('.image-upload-button');

        this.attachEventListeners(uploadButton);
    }

    attachEventListeners(uploadButton) {
        if (!this.fileInput) return;

        this.fileInput.addEventListener('change', (e) => this.handleFileSelect(e.target.files));

        uploadButton.addEventListener('dragover', (e) => {
            e.preventDefault();
            uploadButton.classList.add('drag-over');
        });

        uploadButton.addEventListener('dragleave', () => {
            uploadButton.classList.remove('drag-over');
        });

        uploadButton.addEventListener('drop', (e) => {
            e.preventDefault();
            uploadButton.classList.remove('drag-over');
            this.handleFileSelect(e.dataTransfer.files);
        });
    }

    handleFileSelect(files) {
        this.showError('');
        const newFiles = Array.from(files);

        if (this.images.length + newFiles.length > this.options.maxFiles) {
            this.showError(`Maximum ${this.options.maxFiles} images allowed`);
            return;
        }

        for (const file of newFiles) {
            if (!this.validateFile(file)) {
                continue;
            }

            const reader = new FileReader();
            reader.onload = (e) => {
                this.images.push({
                    file: file,
                    preview: e.target.result
                });
                this.renderPreviews();
                if (this.options.onChange) {
                    this.options.onChange(this.images, this);
                }
            };
            reader.readAsDataURL(file);
        }
    }

    validateFile(file) {
        if (!this.options.acceptedFormats.includes(file.type)) {
            this.showError(`Invalid file type: ${file.name}. Accepted: JPEG, JPG, PNG`);
            return false;
        }

        if (file.size > this.options.maxFileSize) {
            this.showError(`File too large: ${file.name}. Max size: ${this.formatFileSize(this.options.maxFileSize)}`);
            return false;
        }

        return true;
    }

    renderPreviews() {
        if (!this.previewContainer) return;

        if (this.images.length === 0) {
            this.previewContainer.innerHTML = '';
            return;
        }

        this.previewContainer.innerHTML = this.images.map((img, index) => `
            <div class="image-preview-item">
                <img src="${img.preview}" alt="Preview ${index + 1}" />
                <button type="button" class="image-remove-btn" data-index="${index}" title="Remove image">
                    ✕
                </button>
            </div>
        `).join('');

        this.previewContainer.querySelectorAll('.image-remove-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                e.preventDefault();
                const index = parseInt(btn.dataset.index);
                this.removeImage(index);
            });
        });
    }

    removeImage(index) {
        this.images.splice(index, 1);
        this.renderPreviews();
        if (this.options.onChange) {
            this.options.onChange(this.images, this);
        }
    }

    formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
    }

    showError(message) {
        if (!this.errorElement) return;

        if (message) {
            this.errorElement.textContent = message;
            this.errorElement.style.display = 'block';
        } else {
            this.errorElement.style.display = 'none';
        }
    }

    getImages() {
        return this.images;
    }

    getData() {
        return {
            [this.options.name]: this.images.map(img => ({
                name: img.file ? img.file.name : 'image',
                size: img.file ? img.file.size : 0,
                type: img.file ? img.file.type : 'image/png',
                preview: img.preview
            }))
        };
    }

    setData(data) {
        if (data && data[this.options.name] && Array.isArray(data[this.options.name])) {
            this.images = data[this.options.name].map(imgData => ({
                file: null,
                preview: imgData.preview
            }));
            this.renderPreviews();
        }
    }

    clear() {
        this.images = [];
        this.renderPreviews();
        if (this.fileInput) {
            this.fileInput.value = '';
        }
    }

    validate() {
        if (this.options.required && this.images.length === 0) {
            this.showError(`${this.options.label} is required`);
            this.isValid = false;
            return false;
        }
        this.isValid = true;
        return true;
    }

    destroy() {
        if (this.container) {
            this.container.innerHTML = '';
        }
        this.fileInput = null;
        this.previewContainer = null;
        this.errorElement = null;
        this.images = [];
        this.container = null;
    }
}

if (typeof module !== 'undefined' && module.exports) {
    module.exports = ImageUpload;
}