class Toggle {

    constructor(options = {}) {
        this.label = options.label || '';
        this.state = options.initialState !== undefined ? options.initialState : false;
        this.iconOn = options.iconOn || '✓';
        this.iconOff = options.iconOff || '';
        this.onChange = options.onChange || (() => {});
        this.disabled = options.disabled || false;

        this.element = null;
        this.switchElement = null;
        this.inputElement = null;

        this.init();
    }

    init() {
        this.createElement();
        this.attachEventListeners();
        this.updateUI();
    }

    createElement() {

        this.element = document.createElement('div');
        this.element.className = 'toggle-container';
        if (this.disabled) {
            this.element.classList.add('disabled');
        }

        this.inputElement = document.createElement('input');
        this.inputElement.type = 'checkbox';
        this.inputElement.className = 'toggle-input';
        this.inputElement.checked = this.state;
        this.inputElement.disabled = this.disabled;
        this.inputElement.setAttribute('role', 'switch');
        this.inputElement.setAttribute('aria-checked', this.state);

        this.switchElement = document.createElement('div');
        this.switchElement.className = 'toggle-switch';
        if (this.state) {
            this.switchElement.classList.add('active');
        }

        const thumb = document.createElement('div');
        thumb.className = 'toggle-thumb';

        const iconOffElement = document.createElement('span');
        iconOffElement.className = 'toggle-icon toggle-icon-off';
        iconOffElement.textContent = this.iconOff;

        const iconOnElement = document.createElement('span');
        iconOnElement.className = 'toggle-icon toggle-icon-on';
        iconOnElement.textContent = this.iconOn;

        thumb.appendChild(iconOffElement);
        thumb.appendChild(iconOnElement);
        this.switchElement.appendChild(thumb);

        if (this.label) {
            const labelElement = document.createElement('span');
            labelElement.className = 'toggle-label';
            labelElement.textContent = this.label;
            this.element.appendChild(labelElement);
        }

        this.element.appendChild(this.inputElement);
        this.element.appendChild(this.switchElement);
    }

    attachEventListeners() {

        this.element.addEventListener('click', (e) => {
            if (!this.disabled) {
                this.toggle();
            }
        });

        this.element.addEventListener('keydown', (e) => {
            if (!this.disabled && (e.key === ' ' || e.key === 'Enter')) {
                e.preventDefault();
                this.toggle();
            }
        });

        this.element.setAttribute('tabindex', '0');
    }

    toggle() {
        this.setState(!this.state);
    }

    setState(newState) {
        if (this.disabled) return;

        const oldState = this.state;
        this.state = newState;
        this.updateUI();

        if (oldState !== newState) {
            this.onChange(this.state);
        }
    }

    updateUI() {
        if (this.state) {
            this.switchElement.classList.add('active');
        } else {
            this.switchElement.classList.remove('active');
        }

        this.inputElement.checked = this.state;
        this.inputElement.setAttribute('aria-checked', this.state);
    }

    getState() {
        return this.state;
    }

    enable() {
        this.disabled = false;
        this.element.classList.remove('disabled');
        this.inputElement.disabled = false;
    }

    disable() {
        this.disabled = true;
        this.element.classList.add('disabled');
        this.inputElement.disabled = true;
    }

    getElement() {
        return this.element;
    }

    destroy() {
        if (this.element && this.element.parentNode) {
            this.element.parentNode.removeChild(this.element);
        }
        this.element = null;
        this.switchElement = null;
        this.inputElement = null;
    }
}

if (typeof module !== 'undefined' && module.exports) {
    module.exports = Toggle;
}