class FloatingStorageButton {

    constructor(documentGenerator) {
        this.documentGenerator = documentGenerator;
        this.hasStoredData = false;
        this.isUnlocked = false;
        this.dropdown = null;
        this.init();
    }

    init() {
        this.createButton();
        this.updateButtonState();
    }

    createButton() {

        this.container = document.createElement('div');
        this.container.className = 'floating-storage-button';
        this.container.innerHTML = `
            <div class="storage-button-content">
                <button class="storage-btn" type="button">
                    <span class="storage-icon">💾</span>
                    <span class="storage-text">Remember Info</span>
                </button>
            </div>
        `;

        document.body.appendChild(this.container);

        this.button = this.container.querySelector('.storage-btn');
        this.icon = this.container.querySelector('.storage-icon');
        this.text = this.container.querySelector('.storage-text');
        this.content = this.container.querySelector('.storage-button-content');
    }

    updateButtonState() {
        this.hasStoredData = this.documentGenerator.storageDataManager ?
            this.documentGenerator.storageDataManager.hasStoredData() :
            this.documentGenerator.secureStorage.hasStoredData();

        if (this.hasStoredData) {
            this.createDropdownMode();
        } else {
            this.createSingleButtonMode();
            this.isUnlocked = false;
        }
    }

    createSingleButtonMode() {

        if (this.dropdown) {
            this.dropdown = null;
        }

        this.content.innerHTML = `
            <button class="storage-btn single-mode" type="button">
                <span class="storage-icon">💾</span>
                <span class="storage-text">Save for Later</span>
            </button>
        `;

        this.button = this.content.querySelector('.storage-btn');

        this.button.addEventListener('click', () => {
            this.documentGenerator.handleSaveData();
        });
    }

    createDropdownMode() {
        this.content.innerHTML = '<div class="storage-dropdown"></div>';
        const dropdownContainer = this.content.querySelector('.storage-dropdown');

        const dropdownItems = this.isUnlocked ? [

            { text: '💾 Update Data', action: 'save' },
            { text: '🗑️ Clear Saved Data', action: 'clear' }
        ] : [

            { text: '📥 Load Saved Data', action: 'load' },
            { text: '🗑️ Clear Saved Data', action: 'clear' }
        ];

        const defaultAction = this.isUnlocked ? '💾 Update Data' : '📥 Load Saved Data';
        const defaultIndex = 0;

        this.dropdown = new Dropdown(dropdownContainer, {
            items: dropdownItems,
            defaultText: defaultAction,
            splitButton: true,
            onDefaultAction: () => {

                const defaultItem = dropdownItems[defaultIndex];
                switch (defaultItem.action) {
                    case 'load':
                        this.documentGenerator.handleLoadData();
                        break;
                    case 'save':

                        this.documentGenerator.handleSaveData(true);
                        break;
                    case 'clear':
                        this.documentGenerator.handleClearData();
                        break;
                }
            },
            onSelect: (item) => {
                switch (item.action) {
                    case 'load':
                        this.documentGenerator.handleLoadData();
                        break;
                    case 'save':

                        this.documentGenerator.handleSaveData(true);
                        break;
                    case 'clear':
                        this.documentGenerator.handleClearData();
                        break;
                }
            }
        });

        this.dropdown.selectItem(defaultIndex, false);

        const trigger = dropdownContainer.querySelector('.dropdown-trigger');
        const mainButton = dropdownContainer.querySelector('.dropdown-main-button');
        const arrowButton = dropdownContainer.querySelector('.dropdown-arrow-button');

        if (this.isUnlocked) {
            trigger.classList.add('unlocked');
        } else {
            trigger.classList.remove('unlocked');
        }

    }

    refresh() {
        this.updateButtonState();
    }

    markAsUnlocked() {
        this.isUnlocked = true;

        this.hasStoredData = this.documentGenerator.storageDataManager ?
            this.documentGenerator.storageDataManager.hasStoredData() :
            this.documentGenerator.secureStorage.hasStoredData();

        if (this.hasStoredData) {
            this.createDropdownMode();
        }
    }

    resetUnlockedState() {
        this.isUnlocked = false;
        this.updateButtonState();
    }
}