class PhraseModal {
    constructor() {
        this.modal = null;
        this.minPhraseLength = 4;
        this.currentAction = 'save';
    }

    createFormContent(action) {

        const expirationDays = window.SecureStorage ? new window.SecureStorage().getExpirationDays() : 30;

        const description = action === 'save'
            ? `<strong>Create a security phrase to encrypt your information.</strong><br>
               This phrase will be used to encrypt your data before storing it in cookies.
               Remember this phrase - you'll need it to load your information later!<br>
               <small>Saved data will expire in ${expirationDays} days.</small>`
            : `<strong>Enter your security phrase to decrypt your saved information.</strong><br>
               Use the same phrase you used when saving your information.`;

        const helpText = action === 'save'
            ? 'Choose a memorable phrase that\'s at least 4 characters long.'
            : 'Enter the exact phrase you used when saving your data.';

        return `
            <p>${description}</p>
            <div class="form-group">
                <label for="securityPhrase">Security Phrase:</label>
                <input type="password" id="securityPhrase" placeholder="Enter at least ${this.minPhraseLength} characters" autocomplete="off">
                <small id="phraseHelpText">${helpText}</small>
            </div>
            <div id="phraseError" class="error" style="display: none;"></div>
        `;
    }

    setupInputListeners() {
        const phraseInput = this.modal.modal.querySelector('#securityPhrase');
        const confirmBtn = this.modal.modal.querySelector('[data-action="confirm"]');

        if (!phraseInput || !confirmBtn) return;

        phraseInput.addEventListener('input', () => {
            this.validatePhrase();
        });

        phraseInput.addEventListener('keydown', (e) => {
            if (e.key === 'Enter') {
                e.preventDefault();
                if (this.isValidPhrase(phraseInput.value.trim())) {
                    confirmBtn.click();
                }
            }
        });

        this.validatePhrase();

        setTimeout(() => phraseInput.focus(), 100);
    }

    async show(action = 'save') {
        this.currentAction = action;

        this.modal = new Modal({
            title: `${action === 'save' ? 'Save' : 'Load'} Information Securely`,
            content: this.createFormContent(action),
            size: 'medium',
            buttons: [
                {
                    text: 'Cancel',
                    action: 'cancel',
                    variant: 'secondary'
                },
                {
                    text: 'Confirm',
                    action: 'confirm',
                    variant: 'primary',
                    handler: (modal) => {
                        return this.handleConfirm(modal);
                    }
                }
            ],
            onShow: () => {
                this.setupInputListeners();
            }
        });

        const result = await this.modal.show();

        if (result === 'confirm') {
            const phraseInput = this.modal.modal.querySelector('#securityPhrase');
            return phraseInput ? phraseInput.value.trim() : null;
        }

        return null;
    }

    handleConfirm(modal) {
        const phraseInput = modal.modal.querySelector('#securityPhrase');
        const phrase = phraseInput ? phraseInput.value.trim() : '';

        if (!this.isValidPhrase(phrase)) {
            this.showError(`Please enter a phrase that's at least ${this.minPhraseLength} characters long.`);
            if (phraseInput) phraseInput.focus();
            return false;
        }

        this.hideError();
        return true;
    }

    validatePhrase() {
        if (!this.modal || !this.modal.modal) return false;

        const phraseInput = this.modal.modal.querySelector('#securityPhrase');
        const confirmBtn = this.modal.modal.querySelector('[data-action="confirm"]');

        if (!phraseInput || !confirmBtn) return false;

        const phrase = phraseInput.value.trim();
        const isValid = this.isValidPhrase(phrase);

        confirmBtn.disabled = !isValid;

        if (phrase.length > 0) {
            phraseInput.classList.toggle('valid', isValid);
            phraseInput.classList.toggle('invalid', !isValid);
        } else {
            phraseInput.classList.remove('valid', 'invalid');
        }

        return isValid;
    }

    isValidPhrase(phrase) {
        return phrase && phrase.length >= this.minPhraseLength;
    }

    showError(message) {
        if (!this.modal || !this.modal.modal) return;

        const phraseError = this.modal.modal.querySelector('#phraseError');
        const phraseInput = this.modal.modal.querySelector('#securityPhrase');

        if (phraseError) {
            phraseError.textContent = message;
            phraseError.style.display = 'block';
        }

        if (phraseInput) {
            phraseInput.classList.add('invalid');
        }
    }

    hideError() {
        if (!this.modal || !this.modal.modal) return;

        const phraseError = this.modal.modal.querySelector('#phraseError');
        const phraseInput = this.modal.modal.querySelector('#securityPhrase');

        if (phraseError) {
            phraseError.style.display = 'none';
        }

        if (phraseInput) {
            phraseInput.classList.remove('invalid');
        }
    }

    setMinPhraseLength(length) {
        this.minPhraseLength = Math.max(1, length);
    }

    destroy() {
        if (this.modal) {
            this.modal.destroy();
            this.modal = null;
        }
    }
}

window.phraseModal = new PhraseModal();