class StorageUIManager {

    constructor(documentGenerator) {
        this.documentGenerator = documentGenerator;
        this.floatingButton = null;
        this.secureStorage = documentGenerator.secureStorage;
    }

    async initialize() {

        this.initializeFloatingButton();

        this.checkForSavedDataOnLoad();
    }

    initializeFloatingButton() {

        if (typeof FloatingStorageButton !== 'undefined') {
            this.floatingButton = new FloatingStorageButton(this.documentGenerator);
        } else {

            setTimeout(() => this.initializeFloatingButton(), 100);
        }
    }

    refreshStorageUI() {
        if (this.floatingButton) {
            this.floatingButton.refresh();
        }
    }

    async checkForSavedDataOnLoad() {

        setTimeout(async () => {
            const hasStoredData = this.documentGenerator.storageDataManager ?
                this.documentGenerator.storageDataManager.hasStoredData() :
                this.secureStorage.hasStoredData();

            if (hasStoredData) {

                const remainingDays = this.documentGenerator.storageDataManager ?
                    this.documentGenerator.storageDataManager.getRemainingDays() :
                    this.secureStorage.getRemainingDays();

                if (remainingDays !== null && remainingDays <= 0) {
                    this.secureStorage.clearStoredData();
                    if (this.floatingButton) {
                        this.floatingButton.resetUnlockedState();
                    }
                    return;
                }

                const shouldLoad = await this.showSavedDataAlert();
                if (shouldLoad) {

                    this.documentGenerator.handleLoadData();
                }
            }
        }, 500);
    }

    async showSavedDataAlert() {

        const expirationDays = this.documentGenerator.storageDataManager ?
            (this.documentGenerator.storageDataManager.getRemainingDays() || this.documentGenerator.storageDataManager.getExpirationDays()) :
            (this.secureStorage.getRemainingDays() || this.secureStorage.getExpirationDays());

        const content = `
            <p>Saved data will expire in ${expirationDays} days.</p>
            <div class="alert-benefits" style="background: var(--gradient-info-subtle); padding: var(--spacing-lg); border-radius: var(--radius-md); margin: var(--spacing-lg) 0; border-left: 4px solid var(--color-info-border);">
                <div class="benefit-item" style="display: flex; align-items: center; gap: var(--spacing-md); margin-bottom: var(--spacing-md);">
                    <span class="benefit-icon" style="font-size: 1.2rem; width: 20px; text-align: center;">⚡</span>
                    <span>Quick access to your saved data</span>
                </div>
                <div class="benefit-item" style="display: flex; align-items: center; gap: var(--spacing-md); margin-bottom: var(--spacing-md);">
                    <span class="benefit-icon" style="font-size: 1.2rem; width: 20px; text-align: center;">🔐</span>
                    <span>Your information is encrypted and secure</span>
                </div>
                <div class="benefit-item" style="display: flex; align-items: center; gap: var(--spacing-md);">
                    <span class="benefit-icon" style="font-size: 1.2rem; width: 20px; text-align: center;">✏️</span>
                    <span>Continue where you left off</span>
                </div>
            </div>
        `;

        const modal = new Modal({
            title: 'Previous Session Data Discovered',
            content: content,
            size: 'medium',
            buttons: [
                {
                    text: 'Close',
                    action: 'dismiss',
                    variant: 'secondary'
                },
                {
                    text: 'Clear Data',
                    action: 'clear',
                    variant: 'warning',
                    handler: async (modal) => {

                        await this.documentGenerator.handleClearData();
                        return true;
                    }
                },
                {
                    text: 'Load Data',
                    action: 'load',
                    variant: 'primary'
                }
            ]
        });

        const result = await modal.show();

        return result === 'load';
    }

    handleStorageOperation(operation, success) {
        if (!success) return;

        switch (operation) {
            case 'save':

                if (this.floatingButton) {
                    this.floatingButton.resetUnlockedState();
                } else {
                    this.refreshStorageUI();
                }
                break;

            case 'save_and_unlock':

                if (this.floatingButton) {
                    this.floatingButton.markAsUnlocked();
                }
                break;

            case 'update':

                this.refreshStorageUI();
                break;

            case 'load':

                if (this.floatingButton) {
                    this.floatingButton.markAsUnlocked();
                }
                break;

            case 'clear':

                if (this.floatingButton) {
                    this.floatingButton.resetUnlockedState();
                } else {
                    this.refreshStorageUI();
                }
                break;
        }
    }

    getFloatingButton() {
        return this.floatingButton;
    }
}