class DemographicsTab {
    constructor(containerId) {
        this.container = document.getElementById(containerId);
        if (!this.container) {
            console.error(`Demographics tab container with ID "${containerId}" not found`);
            return;
        }

        this.jobTitleInput = null;
        this.jobPurposeInput = null;
        this.otherShiftInfoInput = null;

        this.workWeekCalculator = null;

        this.essentialFunctionsRepeater = null;
        this.marginalFunctionsRepeater = null;
        this.breaksRepeater = null;

        this.imageUpload = null;

        this.render();
        this.init();
    }

    render() {
        this.container.innerHTML = `
            <div class="demographics-content">
                <h2>Overview</h2>
                <p>This section will collect demographic information.</p>

                <div id="job-title-container"></div>
                <div id="job-purpose-container"></div>

                <!-- Essential Functions Repeater -->
                <div id="essential-functions-repeater-container"></div>

                <!-- Marginal Functions Repeater -->
                <div id="marginal-functions-repeater-container"></div>

                <div id="work-week-calculator-container"></div>
                <div id="breaks-container"></div>
                <div id="other-shift-info-container"></div>

                <!-- Image Upload Component -->
                <div id="image-upload-container"></div>

                <!-- Next button container -->
                <div class="form-actions-right">
                    <div id="demographics-next-button-container"></div>
                </div>
            </div>
        `;
    }

    init() {
        this.initializeComponents();
    }

    initializeComponents() {

        if (typeof TextInput === 'undefined' || typeof Repeater === 'undefined' || typeof WorkWeekCalculator === 'undefined' || typeof AreaInput === 'undefined' || typeof ImageUpload === 'undefined') {
            setTimeout(() => this.initializeComponents(), 100);
            return;
        }

        this.initializeTextInputs();
        this.initializeWorkWeekCalculator();
        this.initializeRepeaters();
        this.initializeImageUpload();
    }

    initializeTextInputs() {

        this.jobTitleInput = new TextInput({
            containerId: 'job-title-container',
            id: 'demo-job-title',
            name: 'jobTitle',
            label: 'Job Title',
            placeholder: 'Enter job title',
            required: true
        });

        this.jobPurposeInput = new TextInput({
            containerId: 'job-purpose-container',
            id: 'demo-job-purpose',
            name: 'jobPurpose',
            label: 'Job Purpose',
            placeholder: 'Enter job purpose',
            required: true
        });

        this.otherShiftInfoInput = new TextInput({
            containerId: 'other-shift-info-container',
            id: 'demo-other-shift-info',
            name: 'otherShiftInfo',
            label: 'Other Shift Information',
            placeholder: 'Any additional shift information...',
            required: false,
            helpText: 'Seasonal changes, weekend volume demands, holiday schedules, etc.'
        });
    }

    initializeWorkWeekCalculator() {
        this.workWeekCalculator = new WorkWeekCalculator({
            containerId: 'work-week-calculator-container',
            id: 'demo-work-week',
            name: 'workSchedule',
            label: 'Work Schedule',
            defaultWeeklyHours: 40,
            defaultShiftLength: 8,
            defaultShiftsPerWeek: 5,
            required: false
        });
    }

    initializeRepeaters() {

        this.essentialFunctionsRepeater = new Repeater({
            containerId: 'essential-functions-repeater-container',
            id: 'essential-functions-repeater',
            name: 'essentialFunctions',
            label: 'Essential Functions',
            fields: [
                {
                    name: 'essentialFunction',
                    label: 'Essential Functions',
                    type: 'text',
                    placeholder: 'Primary Tasks',
                    required: true
                },
                {
                    name: 'essentialFunctionDescription',
                    label: 'Short Description',
                    type: 'area',
                    placeholder: 'Task description',
                    required: true,
                    autoGrow: true,
                    rows: 1,
                    maxLength: 150,
                    showCharCounter: true,
                    minHeight: '52px',
                    maxHeight: '150px',
                    validation: (value) => {
                        if (value.length < 10) {
                            return 'Description must be at least 10 characters';
                        }
                        if (value.length > 150) {
                            return 'Description must be 150 characters or less';
                        }
                        return true;
                    }
                }
            ],
            required: true,
            defaultRows: 1,
            showFieldLabels: false
        });

        this.marginalFunctionsRepeater = new Repeater({
            containerId: 'marginal-functions-repeater-container',
            id: 'marginal-functions-repeater',
            name: 'marginalFunctions',
            label: 'Marginal Functions',
            fields: [
                {
                    name: 'marginalFunction',
                    label: 'Marginal Functions',
                    type: 'text',
                    placeholder: 'Secondary Tasks',
                    required: true
                },
                {
                    name: 'marginalFunctionDescription',
                    label: 'Short Description',
                    type: 'area',
                    placeholder: 'Task description',
                    required: true,
                    autoGrow: true,
                    rows: 1,
                    maxLength: 150,
                    showCharCounter: true,
                    minHeight: '52px',
                    maxHeight: '150px',
                    validation: (value) => {
                        if (value.length < 10) {
                            return 'Description must be at least 10 characters';
                        }
                        if (value.length > 150) {
                            return 'Description must be 150 characters or less';
                        }
                        return true;
                    }
                }
            ],
            required: true,
            defaultRows: 1,
            showFieldLabels: false
        });

        this.breaksRepeater = new Repeater({
            containerId: 'breaks-container',
            id: 'breaks-repeater',
            name: 'breaks',
            label: 'Breaks',
            fields: [
                {
                    name: 'breakDescription',
                    label: 'Description',
                    type: 'text',
                    placeholder: 'Break information',
                    required: true
                }
            ],
            required: true,
            defaultRows: 1,
            showFieldLabels: false
        });

        setTimeout(() => this.setDefaultBreaks(), 0);
    }

    initializeImageUpload() {
        this.imageUpload = new ImageUpload({
            containerId: 'image-upload-container',
            id: 'demo-job-images',
            name: 'jobImages',
            label: 'Job Images',
            required: false,
            maxFiles: 12,
            maxFileSize: 5 * 1024 * 1024,
            acceptedFormats: ['image/jpeg', 'image/png', 'image/jpg'],
            helpText: 'Upload images related to the job (optional)'
        });
    }

    setDefaultBreaks() {
        if (this.breaksRepeater) {
            this.breaksRepeater.setData({
                breaks: [
                    { breakDescription: 'One 15-minute break at about 10am' },
                    { breakDescription: 'One 30-minute break at about 12pm' },
                    { breakDescription: 'One 15-minute break at about 2pm' }
                ]
            });
        }
    }

    getData() {
        return {
            ...this.jobTitleInput?.getData(),
            ...this.jobPurposeInput?.getData(),
            ...this.essentialFunctionsRepeater?.getData(),
            ...this.marginalFunctionsRepeater?.getData(),
            ...this.workWeekCalculator?.getData(),
            ...this.breaksRepeater?.getData(),
            ...this.otherShiftInfoInput?.getData(),
            ...this.imageUpload?.getData()
        };
    }

    setData(data) {
        this.jobTitleInput?.setData(data);
        this.jobPurposeInput?.setData(data);
        this.essentialFunctionsRepeater?.setData(data);
        this.marginalFunctionsRepeater?.setData(data);
        this.workWeekCalculator?.setData(data);
        this.breaksRepeater?.setData(data);
        this.otherShiftInfoInput?.setData(data);
        this.imageUpload?.setData(data);
    }

    validate() {
        let isValid = true;

        if (this.jobTitleInput && !this.jobTitleInput.validate()) {
            isValid = false;
        }

        if (this.jobPurposeInput && !this.jobPurposeInput.validate()) {
            isValid = false;
        }

        if (this.essentialFunctionsRepeater && !this.essentialFunctionsRepeater.validate()) {
            isValid = false;
        }

        if (this.marginalFunctionsRepeater && !this.marginalFunctionsRepeater.validate()) {
            isValid = false;
        }

        if (this.breaksRepeater && !this.breaksRepeater.validate()) {
            isValid = false;
        }

        return isValid;
    }
}

if (typeof module !== 'undefined' && module.exports) {
    module.exports = DemographicsTab;
}