class EnvironmentalDemands {
    constructor() {
        this.table = null;
        this.containerId = 'environmental-demands-table-container';
    }

    render() {
        return `
            <div class="demands-section environmental-demands">
                <div id="${this.containerId}"></div>
            </div>
        `;
    }

    init() {

        const container = document.getElementById(this.containerId);
        if (!container) {
            console.warn(`Environmental demands container ${this.containerId} not found, will retry...`);
            return;
        }

        const headerRows = [
            'Wet, humid, or slippery surfaces',
            'Proximity to moving mechanical parts or machinery',
            'Working at heights',
            'Fumes, odors, dust, or airborne particles',
            'Hazardous chemicals (toxic or caustic)',
            'Extreme temperatures (hot or cold, weather-related or non-weather)',
            'High noise levels requiring hearing protection',
            'Hand-arm vibration (e.g., from power tools)',
            'Whole-body vibration (e.g., from vehicles or platforms)',
            'Electrical hazards',
            'Radiation exposure (ionizing or non-ionizing)',
            'Poor lighting or illumination',
            'Confined spaces',
            'Biological hazards (e.g., pathogens or allergens)'
        ];

        this.table = new Table({
            containerId: this.containerId,
            headerColumns: [
                { responsive: { full: 'Not Applicable\n0%', short: 'N/A', letter: 'N' } },
                { responsive: { full: 'Occasional\n1-33%', short: 'Occasional', letter: 'O' } },
                { responsive: { full: 'Frequent\n34-66%', short: 'Frequent', letter: 'F' } },
                { responsive: { full: 'Constant\n67-100%', short: 'Constant', letter: 'C' } },
                { responsive: { full: 'Objective Measurements\n& General Comments', short: 'Comments', letter: 'Notes' } }
            ],
            headerRows: headerRows,
            cellType: 'selectable',
            selectionMode: 'single',
            rowHeaderWidth: '200px',
            columnWidths: ['auto', 'auto', 'auto', 'auto', 'auto'],
            columnTypes: ['selectable', 'selectable', 'selectable', 'selectable', 'input'],
            striped: true,
            hoverable: true,
            showValidationErrors: true,
            onChange: (data) => {

            }
        });
    }

    getData() {
        if (!this.table) {
            console.warn('Environmental demands table not initialized yet, returning empty data');
            return {};
        }
        return this.table.getData();
    }

    setData(data) {
        if (!data) return;

        if (!this.table) {
            console.warn('Environmental demands table not initialized yet, attempting to initialize...');
            this.init();
        }

        if (this.table) {
            this.table.setData(data);
        } else {
            console.error('Failed to initialize environmental demands table for setData');
        }
    }

    clear() {
        if (this.table) {
            this.table.clear();
        }
    }

    validate() {
        if (!this.table) {
            return false;
        }
        return this.table.validate();
    }

    getValidationErrors() {
        if (!this.table) {
            return {};
        }
        return this.table.getValidationErrors();
    }

    clearValidationErrors() {
        if (this.table) {
            this.table.clearValidationErrors();
        }
    }
}

if (typeof module !== 'undefined' && module.exports) {
    module.exports = EnvironmentalDemands;
}