class JobsTab {
    constructor(containerId) {
        this.container = document.getElementById(containerId);
        if (!this.container) {
            console.error(`Jobs tab container with ID "${containerId}" not found`);
            return;
        }
        this.subNav = null;

        this.physicalDemands = new PhysicalDemands();
        this.mobilityDemands = new MobilityDemands();
        this.cognitiveSensoryDemands = new CognitiveSensoryDemands();
        this.environmentalDemands = new EnvironmentalDemands();
        this.liftingPushingPulling = new LiftingPushingPulling();
        this.classificationOfWork = new ClassificationOfWork();

        this.render();
        this.init();
    }

    render() {
        this.container.innerHTML = `
            <div class="jobs-content">
                <!-- SubNav container -->
                <div id="jobs-subnav-container"></div>

                <!-- Navigation buttons container -->
                <div class="form-actions-between">
                    <div id="jobs-back-button-container"></div>
                    <div id="jobs-next-button-container"></div>
                </div>
            </div>
        `;
    }

    init() {

        this.subNav = new SubNav({
            containerId: 'jobs-subnav-container',
            sections: [
                {
                    id: 'physical-demands',
                    title: 'Physical Demands',
                    icon: '💪',
                    hideTitle: true,
                    noPadding: true,
                    content: () => this.physicalDemands.render()
                },
                {
                    id: 'mobility-demands',
                    title: 'Mobility Demands',
                    icon: '🚶',
                    hideTitle: true,
                    noPadding: true,
                    content: () => this.mobilityDemands.render()
                },
                {
                    id: 'cognitive-sensory',
                    title: 'Cognitive & Sensory',
                    icon: '🧠',
                    hideTitle: true,
                    noPadding: true,
                    content: () => this.cognitiveSensoryDemands.render()
                },
                {
                    id: 'environmental',
                    title: 'Environmental Demands',
                    icon: '🌍',
                    hideTitle: true,
                    noPadding: true,
                    content: () => this.environmentalDemands.render()
                },
                {
                    id: 'lifting-pushing-pulling',
                    title: 'Lifting/Pushing/Pulling',
                    icon: '📦',
                    hideTitle: true,
                    noPadding: true,
                    content: () => this.liftingPushingPulling.render()
                },
                {
                    id: 'classification',
                    title: 'Classification of Work',
                    icon: '📋',
                    hideTitle: true,
                    noPadding: true,
                    content: () => this.classificationOfWork.render()
                }
            ],
            defaultSection: 'physical-demands'
        });

        setTimeout(() => {
            this.physicalDemands.init();
            this.mobilityDemands.init();
            this.cognitiveSensoryDemands.init();
            this.environmentalDemands.init();
            this.liftingPushingPulling.init();
            this.classificationOfWork.init();
        }, 100);

        const subNavContainer = document.getElementById('jobs-subnav-container');
        if (subNavContainer) {
            subNavContainer.addEventListener('sectionchange', (e) => {
            });
        }
    }

    ensureInitialized() {
        if (!this.physicalDemands.table) {
            this.physicalDemands.init();
        }
    }

    getData() {

        this.ensureInitialized();

        const data = {
            activeSection: this.subNav?.getActiveSection() || null,
            physicalDemands: this.physicalDemands.getData(),
            mobilityDemands: this.mobilityDemands.getData(),
            cognitiveSensoryDemands: this.cognitiveSensoryDemands.getData(),
            environmentalDemands: this.environmentalDemands.getData(),
            liftingPushingPulling: this.liftingPushingPulling.getData(),
            classificationOfWork: this.classificationOfWork.getData()
        };

        return data;
    }

    setData(data) {
        if (!data) return;

        this.ensureInitialized();

        if (data.activeSection && this.subNav) {
            this.subNav.setActiveSection(data.activeSection);
        }

        if (data.physicalDemands) {
            this.physicalDemands.setData(data.physicalDemands);
        }
        if (data.mobilityDemands) {
            this.mobilityDemands.setData(data.mobilityDemands);
        }
        if (data.cognitiveSensoryDemands) {
            this.cognitiveSensoryDemands.setData(data.cognitiveSensoryDemands);
        }
        if (data.environmentalDemands) {
            this.environmentalDemands.setData(data.environmentalDemands);
        }
        if (data.liftingPushingPulling) {
            this.liftingPushingPulling.setData(data.liftingPushingPulling);
        }
        if (data.classificationOfWork) {
            this.classificationOfWork.setData(data.classificationOfWork);
        }
    }

    validate() {

        const validations = [
            this.physicalDemands.validate(),
            this.mobilityDemands.validate(),
            this.environmentalDemands.validate(),
            this.liftingPushingPulling.validate(),
            this.classificationOfWork.validate()
        ];

        return validations.every(v => v === true);
    }

    getFirstSubnavSectionWithErrors() {

        const sectionOrder = [
            { id: 'physical-demands', validator: () => this.physicalDemands.validate() },
            { id: 'mobility-demands', validator: () => this.mobilityDemands.validate() },
            { id: 'cognitive-sensory', validator: () => this.cognitiveSensoryDemands.validate() },
            { id: 'environmental', validator: () => this.environmentalDemands.validate() },
            { id: 'lifting-pushing-pulling', validator: () => this.liftingPushingPulling.validate() },
            { id: 'classification', validator: () => this.classificationOfWork.validate() }
        ];

        for (const section of sectionOrder) {
            if (!section.validator()) {
                return section.id;
            }
        }

        return null;
    }
}

if (typeof module !== 'undefined' && module.exports) {
    module.exports = JobsTab;
}