class SummaryTab {
    constructor(containerId) {
        this.container = document.getElementById(containerId);
        if (!this.container) {
            console.error(`Summary tab container with ID "${containerId}" not found`);
            return;
        }

        this.summaryInput = null;
        this.generateButton = null;

        this.render();
        this.init();
    }

    render() {
        this.container.innerHTML = `
            <div class="summary-content">
                <h2>Summary</h2>
                <p>This section will provide a summary or additional information.</p>

                <!-- Summary rich text area -->
                <div id="summary-text-container"></div>

                <!-- Generate button container -->
                <div class="form-actions" id="generate-button-container">
                    <!-- Generate button will be created here by the Button component -->
                </div>
            </div>
        `;
    }

    init() {

        this.summaryInput = new AreaInput({
            containerId: 'summary-text-container',
            id: 'summaryText',
            name: 'summaryText',
            label: 'Summary',
            placeholder: 'Enter a summary or additional notes...',
            useRichText: true,
            resize: 'vertical',
            minHeight: '250px',
            maxHeight: '600px',
            required: true,
            minLength: 10,
            maxLength: 5000,
            showCharCounter: true,
            validation: (value) => {
                if (!value || value.trim().length === 0) {
                    return 'Summary is required';
                }
                if (value.length < 10) {
                    return 'Summary must be at least 10 characters';
                }
                if (value.length > 5000) {
                    return 'Summary cannot exceed 5,000 characters';
                }
                return true;
            },
            quillConfig: {
                theme: 'snow',
                modules: {
                    toolbar: [
                        [{ 'header': [1, 2, 3, false] }],
                        ['bold', 'italic', 'underline'],
                        [{ 'list': 'ordered'}, { 'list': 'bullet' }],
                        ['link'],
                        ['clean']
                    ]
                },
                placeholder: 'Enter a summary or additional notes...'
            }
        });

        this.createGenerateButton();
    }

    createGenerateButton() {

        if (typeof Button === 'undefined') {
            console.error('Button component not loaded. Make sure to include button.js');
            return;
        }

        this.generateButton = new Button({
            containerId: 'generate-button-container',
            id: 'generateBtn',
            text: 'Generate Document',
            variant: 'success',
            size: 'medium',
            onClick: (_, __) => {
            }
        });
    }

    setGenerateButtonHandler(handler) {
        if (this.generateButton && typeof handler === 'function') {
            this.generateButton.options.onClick = (_, __) => {
                handler(_, __);
            };
        }
    }

    getData() {
        return {
            summaryText: this.summaryInput ? this.summaryInput.getValue() : ''
        };
    }

    setData(data) {
        if (data.summaryText && this.summaryInput) {
            this.summaryInput.setValue(data.summaryText);
        }
    }

    validate() {
        if (!this.summaryInput) {
            return false;
        }

        return this.summaryInput.validateInput();
    }

    destroy() {
        if (this.summaryInput) {
            this.summaryInput.destroy();
            this.summaryInput = null;
        }
    }
}

if (typeof module !== 'undefined' && module.exports) {
    module.exports = SummaryTab;
}