class TabsManager {
    constructor() {
        this.tabs = null;
        this.introTab = null;
        this.demographicsTab = null;
        this.jobsTab = null;
        this.summaryTab = null;
        this.previewTab = null;
        this.nextButtonManager = null;
    }

    init() {

        this.tabs = new Tabs('documentTabs');

        this.introTab = new IntroTab('tab-intro');
        this.demographicsTab = new DemographicsTab('tab-demographics');
        this.jobsTab = new JobsTab('tab-jobs');
        this.summaryTab = new SummaryTab('tab-summary');
        this.previewTab = new PreviewTab('tab-preview', this);

        const tabsContainer = document.getElementById('documentTabs');
        if (tabsContainer) {
            tabsContainer.addEventListener('tabchange', (e) => {
                this.onTabChange(e.detail);
            });
        }

        this.initializeNextButtons();
    }

    initializeNextButtons() {

        if (typeof NextButtonManager === 'undefined') {
            setTimeout(() => this.initializeNextButtons(), 100);
            return;
        }

        this.nextButtonManager = new NextButtonManager(this);

        this.nextButtonManager.addNextButton('intro', 'intro-next-button-container');
        this.nextButtonManager.addNextButton('demographics', 'demographics-next-button-container');
        this.nextButtonManager.addNextButton('jobs', 'jobs-next-button-container');
        this.nextButtonManager.addNextButton('summary', 'summary-next-button-container');

        this.nextButtonManager.addBackButton('jobs-back-button-container');
    }

    onTabChange(detail) {

        if (detail.tabId === 'tab-preview') {
            this.previewTab.updatePreview();
        }
    }

    getAllData() {
        return {
            intro: this.introTab.getData(),
            demographics: this.demographicsTab.getData(),
            jobs: this.jobsTab.getData(),
            summary: this.summaryTab.getData()
        };
    }

    setAllData(data) {
        if (data.intro) this.introTab.setData(data.intro);
        if (data.demographics) this.demographicsTab.setData(data.demographics);
        if (data.jobs) this.jobsTab.setData(data.jobs);
        if (data.summary) this.summaryTab.setData(data.summary);
    }

    validateAll() {
        const errors = [];
        const tabErrors = [];

        if (!this.introTab.validate()) {
            errors.push('Intro tab: Title and Author are required');
            tabErrors.push({ tab: 'intro', message: 'Title and Author are required' });
        }

        if (!this.demographicsTab.validate()) {
            errors.push('Demographics tab: Name is required');
            tabErrors.push({ tab: 'demographics', message: 'Name is required' });
        }

        if (!this.jobsTab.validate()) {
            errors.push('Jobs tab: At least one job entry is required');
            tabErrors.push({ tab: 'jobs', message: 'At least one job entry is required' });
        }

        if (!this.summaryTab.validate()) {
            errors.push('Summary tab: Summary is required (10-5,000 characters)');
            tabErrors.push({ tab: 'summary', message: 'Summary is required (10-5,000 characters)' });
        }

        return {
            isValid: errors.length === 0,
            errors: errors,
            tabErrors: tabErrors
        };
    }

    getFirstTabWithErrors() {
        const validation = this.validateAll();

        if (validation.isValid) {
            return null;
        }

        const tabOrder = ['intro', 'demographics', 'jobs', 'summary'];

        for (const tabName of tabOrder) {
            if (validation.tabErrors.some(error => error.tab === tabName)) {
                return tabName;
            }
        }

        return null;
    }

    switchToTab(tabName) {
        const tabIds = {
            'intro': 'tab-intro',
            'demographics': 'tab-demographics',
            'jobs': 'tab-jobs',
            'summary': 'tab-summary',
            'preview': 'tab-preview'
        };

        const tabId = tabIds[tabName];
        if (tabId && this.tabs) {
            this.tabs.switchToTabById(tabId);
        }
    }
}

if (typeof module !== 'undefined' && module.exports) {
    module.exports = TabsManager;
}