class PDFUtils {

    static COLORS = {
        primary: [41, 128, 185],
        secondary: [52, 73, 94],
        accent: [231, 76, 60],
        success: [39, 174, 96],
        warning: [241, 196, 15],
        light: [236, 240, 241],
        dark: [44, 62, 80],
        text: [0, 0, 0],
        lightText: [127, 140, 141]
    };

    static FONT_SIZES = {
        title: 24,
        heading1: 16,
        heading2: 12,
        heading3: 10,
        body: 10,
        small: 9,
        tiny: 8
    };

    static SPACING = {
        margin: 15,
        sectionGap: 10,
        lineHeight: 7,
        fieldGap: 5
    };

    static formatDate(date) {
        if (!date) return new Date().toLocaleDateString();
        if (typeof date === 'string') return date;
        if (date instanceof Date) return date.toLocaleDateString();
        return new Date().toLocaleDateString();
    }

    static formatAddress(addressData) {
        const parts = [];
        if (addressData?.street) parts.push(addressData.street);
        if (addressData?.city) parts.push(addressData.city);
        if (addressData?.state) parts.push(addressData.state);
        if (addressData?.zip) parts.push(addressData.zip);
        return parts.join(', ') || 'Not specified';
    }

    static formatClassification(classification) {
        if (!classification) return 'Not specified';
        const parts = [];
        if (classification.physicalLevel) parts.push(`Physical: ${classification.physicalLevel}`);
        if (classification.cognitiveLevel) parts.push(`Cognitive: ${classification.cognitiveLevel}`);
        return parts.length > 0 ? parts.join(', ') : 'Not specified';
    }

    static truncate(text, maxLength = 100) {
        if (!text) return '';
        if (text.length <= maxLength) return text;
        return text.substring(0, maxLength) + '...';
    }

    static isEmpty(obj) {
        if (!obj) return true;
        if (typeof obj !== 'object') return false;
        return Object.keys(obj).length === 0;
    }

    static escape(text) {
        if (!text) return '';
        return String(text)
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#039;');
    }

    static getDisplayValue(value, defaultText = 'Not specified') {
        if (value === null || value === undefined || value === '') return defaultText;
        return String(value);
    }

    static arrayToObject(array) {
        if (!Array.isArray(array)) return {};
        const obj = {};
        array.forEach((item, index) => {
            if (typeof item === 'object') {
                Object.assign(obj, item);
            } else {
                obj[`item_${index}`] = item;
            }
        });
        return obj;
    }

    static hexToRgb(hexColor) {
        if (!hexColor) return [0, 0, 0];

        const hex = hexColor.replace('#', '');

        const fullHex = hex.length === 3
            ? hex.split('').map(char => char + char).join('')
            : hex;

        const r = parseInt(fullHex.substring(0, 2), 16);
        const g = parseInt(fullHex.substring(2, 4), 16);
        const b = parseInt(fullHex.substring(4, 6), 16);

        if (isNaN(r) || isNaN(g) || isNaN(b)) {
            return [0, 0, 0];
        }

        return [r, g, b];
    }
}